USE [sandbox]
GO

-- Generate 1,000,000 rows of random data for NoSparseColumnTest table
-- Using CTE with CROSS JOIN for efficient large dataset generation

SET NOCOUNT ON;

-- Clear existing data (optional - remove if you want to keep existing data)
-- TRUNCATE TABLE [dbo].[NoSparseColumnTest];

-- Generate random seed for consistent randomization in this session
DECLARE @Seed INT = ABS(CHECKSUM(NEWID()));

WITH 
-- Generate base numbers using a simpler approach
Level1 AS (
    SELECT TOP 1000 object_id as n
    FROM sys.all_objects
),
-- Cross join to get 1,000,000 combinations (1000 x 1000)  
Numbers AS (
    SELECT TOP 1000000
        ROW_NUMBER() OVER (ORDER BY l1.n, l2.n) AS RowNum
    FROM Level1 l1
    CROSS JOIN Level1 l2
),
-- Generate the random data
RandomData AS (
    SELECT 
        RowNum AS ID,
        -- CustomerID: Random integer between 1 and 50,000, with some NULLs (5%)
        CASE WHEN ABS(CHECKSUM(NEWID())) % 100 < 5 THEN NULL 
             ELSE ABS(CHECKSUM(NEWID())) % 50000 + 1 
        END AS CustomerID,
        
        -- TrackingDate: Random dates between 2020-01-01 and 2025-12-31, with some NULLs (3%)
        CASE WHEN ABS(CHECKSUM(NEWID())) % 100 < 3 THEN NULL
             ELSE DATEADD(DAY, 
                         ABS(CHECKSUM(NEWID())) % DATEDIFF(DAY, '2020-01-01', '2025-12-31'), 
                         '2020-01-01')
        END AS TrackingDate,
        
        -- SomeFlag: Random tinyint 0, 1, 2, or NULL (with NULLs being 10% of records)
        CASE WHEN ABS(CHECKSUM(NEWID())) % 100 < 10 THEN NULL
             ELSE ABS(CHECKSUM(NEWID())) % 3
        END AS SomeFlag,
        
        -- aNumber: Random numeric values between -999999.9999 and 999999.9999, with some NULLs (7%)
        CASE WHEN ABS(CHECKSUM(NEWID())) % 100 < 7 THEN NULL
             ELSE CAST((ABS(CHECKSUM(NEWID())) % 2000000 - 1000000) + 
                      (ABS(CHECKSUM(NEWID())) % 10000) / 10000.0 AS NUMERIC(38,4))
        END AS aNumber
    FROM Numbers
)

-- Insert the data in batches to avoid transaction log issues
INSERT INTO [dbo].[SparseColumnTest] ([ID], [CustomerID], [TrackingDate], [SomeFlag], [aNumber])
SELECT ID, CustomerID, null, null, null
FROM RandomData;

-- Display summary statistics
SELECT 
    COUNT(*) as TotalRows,
    COUNT([CustomerID]) as NonNullCustomerIDs,
    COUNT([TrackingDate]) as NonNullTrackingDates,
    COUNT([SomeFlag]) as NonNullSomeFlags,
    COUNT([aNumber]) as NonNullaNumbers,
    MIN([TrackingDate]) as EarliestDate,
    MAX([TrackingDate]) as LatestDate,
    MIN([aNumber]) as MinNumber,
    MAX([aNumber]) as MaxNumber
FROM [dbo].[NoSparseColumnTest];

PRINT 'Data generation completed successfully!';
PRINT CONCAT('Generated ', @@ROWCOUNT, ' rows of random data.');

SET NOCOUNT OFF;