USE [sandbox]
GO

SET NOCOUNT ON;

-- First, let's check how many NULLs we have in each column
SELECT 
    'Before Update' AS Status,
    SUM(CASE WHEN CustomerID IS NULL THEN 1 ELSE 0 END) AS CustomerID_NULLs,
    SUM(CASE WHEN TrackingDate IS NULL THEN 1 ELSE 0 END) AS TrackingDate_NULLs,
    SUM(CASE WHEN SomeFlag IS NULL THEN 1 ELSE 0 END) AS SomeFlag_NULLs,
    SUM(CASE WHEN aNumber IS NULL THEN 1 ELSE 0 END) AS aNumber_NULLs,
    COUNT(*) AS TotalRows
FROM [dbo].[SparseColumnTest];

-- Update CustomerID: 10% of NULLs to random values between 1 and 50000
WITH CustomerIDNulls AS (
    SELECT TOP (10) PERCENT ID
    FROM [dbo].[SparseColumnTest]
    WHERE CustomerID IS NULL
    ORDER BY NEWID()  -- Random selection
)
UPDATE nsc
SET CustomerID = ABS(CHECKSUM(NEWID())) % 50000 + 1
FROM [dbo].[SparseColumnTest] nsc
INNER JOIN CustomerIDNulls c ON nsc.ID = c.ID;

PRINT CONCAT('Updated ', @@ROWCOUNT, ' CustomerID NULL values');

-- Update TrackingDate: 10% of NULLs to random dates between 2020-01-01 and 2025-12-31
WITH TrackingDateNulls AS (
    SELECT TOP (10) PERCENT ID
    FROM [dbo].[SparseColumnTest]
    WHERE TrackingDate IS NULL
    ORDER BY NEWID()  -- Random selection
)
UPDATE nsc
SET TrackingDate = DATEADD(DAY, 
                          ABS(CHECKSUM(NEWID())) % DATEDIFF(DAY, '2020-01-01', '2025-12-31'), 
                          '2020-01-01')
FROM [dbo].[SparseColumnTest] nsc
INNER JOIN TrackingDateNulls t ON nsc.ID = t.ID;

PRINT CONCAT('Updated ', @@ROWCOUNT, ' TrackingDate NULL values');

-- Update SomeFlag: 10% of NULLs to random values (0, 1, or 2)
WITH SomeFlagNulls AS (
    SELECT TOP (10) PERCENT ID
    FROM [dbo].[SparseColumnTest]
    WHERE SomeFlag IS NULL
    ORDER BY NEWID()  -- Random selection
)
UPDATE nsc
SET SomeFlag = ABS(CHECKSUM(NEWID())) % 3
FROM [dbo].[SparseColumnTest] nsc
INNER JOIN SomeFlagNulls s ON nsc.ID = s.ID;

PRINT CONCAT('Updated ', @@ROWCOUNT, ' SomeFlag NULL values');

-- Update aNumber: 10% of NULLs to random numeric values between -999999.9999 and 999999.9999
WITH aNumberNulls AS (
    SELECT TOP (10) PERCENT ID
    FROM [dbo].[SparseColumnTest]
    WHERE aNumber IS NULL
    ORDER BY NEWID()  -- Random selection
)
UPDATE nsc
SET aNumber = CAST((ABS(CHECKSUM(NEWID())) % 2000000 - 1000000) + 
                   (ABS(CHECKSUM(NEWID())) % 10000) / 10000.0 AS NUMERIC(38,4))
FROM [dbo].[SparseColumnTest] nsc
INNER JOIN aNumberNulls a ON nsc.ID = a.ID;

PRINT CONCAT('Updated ', @@ROWCOUNT, ' aNumber NULL values');

-- Show the results after update
SELECT 
    'After Update' AS Status,
    SUM(CASE WHEN CustomerID IS NULL THEN 1 ELSE 0 END) AS CustomerID_NULLs,
    SUM(CASE WHEN TrackingDate IS NULL THEN 1 ELSE 0 END) AS TrackingDate_NULLs,
    SUM(CASE WHEN SomeFlag IS NULL THEN 1 ELSE 0 END) AS SomeFlag_NULLs,
    SUM(CASE WHEN aNumber IS NULL THEN 1 ELSE 0 END) AS aNumber_NULLs,
    COUNT(*) AS TotalRows
FROM [dbo].[SparseColumnTest];

-- Sample of updated data
SELECT TOP 20 *
FROM [dbo].[SparseColumnTest]
WHERE CustomerID IS NOT NULL 
   OR TrackingDate IS NOT NULL 
   OR SomeFlag IS NOT NULL 
   OR aNumber IS NOT NULL
ORDER BY ID;

PRINT 'Update completed successfully!';

SET NOCOUNT OFF;